const express = require('express');
const router = express.Router();
const mongoose = require('mongoose');
const Exam = require('../../models/teacher/Exam');
const Question = require('../../models/teacher/Question');
const Submission = require('../../models/student/Submission');

// Get active/live exams available for students
router.get('/active', async (req, res) => {
  try {
    const { classId } = req.query;
    const filter = { status: 'live' };
    
    if (classId) filter.classId = classId;
    
    const exams = await Exam.find(filter);
    res.json({ success: true, exams });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get exam details for taking exam (without correct answers)
router.get('/:id', async (req, res) => {
  try {
    const exam = await Exam.findById(req.params.id);
    if (!exam) {
      return res.status(404).json({ success: false, error: 'Exam not found' });
    }
    
    if (exam.status !== 'live') {
      return res.status(400).json({ success: false, error: 'Exam is not currently live' });
    }
    
    // Get questions but hide correct answers
    const questions = await Question.find({ examId: exam._id }).sort({ order: 1 });
    const sanitizedQuestions = questions.map(q => ({
      _id: q._id,
      questionText: q.questionText,
      options: q.options.map(opt => ({
        optionId: opt.optionId,
        text: opt.text
        // isCorrect is intentionally omitted
      })),
      marks: q.marks,
      order: q.order
    }));
    
    res.json({ success: true, exam, questions: sanitizedQuestions });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Submit exam answers
router.post('/:id/submit', async (req, res) => {
  try {
    const { studentId, answers } = req.body;
    const examId = req.params.id;
    
    const exam = await Exam.findById(examId);
    if (!exam || exam.status !== 'live') {
      return res.status(400).json({ success: false, error: 'Exam is not available for submission' });
    }
    
    // Check if student has already submitted
    const existingSubmission = await Submission.findOne({ examId, studentId });
    if (existingSubmission) {
      return res.status(400).json({ success: false, error: 'Exam already submitted' });
    }
    
    // Get questions to calculate score
    const questions = await Question.find({ examId });
    let score = 0;
    
    const submissionAnswers = answers.map(answer => {
      const question = questions.find(q => q._id.toString() === answer.questionId);
      if (question) {
        const correctOption = question.options.find(opt => opt.isCorrect);
        const isCorrect = correctOption && correctOption.optionId === answer.selectedOption;
        if (isCorrect) {
          score += question.marks;
        }
        return {
          questionId: answer.questionId,
          selectedOptionId: answer.selectedOption,
          isCorrect,
          marksObtained: isCorrect ? question.marks : 0
        };
      }
      return answer;
    });
    
    const submission = new Submission({
      examId,
      studentId,
      answers: submissionAnswers,
      totalScore: score,
      percentage: exam.totalMarks > 0 ? (score / exam.totalMarks) * 100 : 0,
      passed: score >= exam.passingMarks,
      status: 'submitted',
      submittedAt: new Date()
    });
    
    await submission.save();
    
    // Emit socket event for real-time updates
    req.app.get('io').emit('examSubmission', { examId, studentId, score });
    
    res.json({ success: true, submission, score: submission.totalScore });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get student's exam history
router.get('/history/:studentId', async (req, res) => {
  try {
    const { studentId } = req.params;
    
    const submissions = await Submission.find({ studentId })
      .populate({
        path: 'examId',
        model: 'Exam',
        select: 'title description totalMarks passingMarks duration'
      })
      .sort({ submittedAt: -1 });
    
    res.json({ success: true, submissions });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get specific submission details
router.get('/submission/:submissionId', async (req, res) => {
  try {
    const submission = await Submission.findById(req.params.submissionId)
      .populate({
        path: 'examId',
        model: 'Exam',
        select: 'title description totalMarks passingMarks'
      });
    
    if (!submission) {
      return res.status(404).json({ success: false, error: 'Submission not found' });
    }
    
    res.json({ success: true, submission });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

module.exports = router;